define([
    'backbone',
    'moment',
    'Layout',
    'modules/appointments/booked/appointment-cancel-reason-section-view',
    'modules/appointments/booked/booked-appointment-view',
    'modules/appointments/booked/booked-appointments-collection',
], function(
    Backbone,
    moment,
    Layout,
    AppointmentCancelReasonSectionView,
    BookedAppointmentView,
    BookedAppointmentsCollection
) {
    'use strict';

    describe('Booked Appointment View', function() {
        var view;
        var $view;
        var ccview;
        var $ccview;
        var model;

        var layout = new Layout();

        layout.render();

        beforeEach(function() {
            // TODO this model should be imported
            // eslint-disable-next-line new-cap
            model = new BookedAppointmentsCollection.prototype.model({
                'appointmentLength': '30',
                'appointmentTime': '06/11/2016 09:00:00',
                'bookingNote': 'Reason',
                'clinic': { 'name': 'ABC Clinic', 'friendlyName': 'FriendlyName', 'askForCheckIn': false},
                'currentStatus': 'FUTURE',
                'link': [
                    {
                        'href': 'http://IP      /VeteranAppointmentRequestService/rest/direct-scheduling/site/688/patient/EDIPI/0000000001/cancel-appointment',
                        'rel': 'related',
                        'title': 'cancel-appointment',
                    },
                ],
                'purpose': 'Medication Concern',
                'type': 'REGULAR',
                'siteCode': '244',
                'facilityName': 'FACILITY VAMC',
            });

            view = new BookedAppointmentView({model: model});

            layout.showChildView('content', view);
            $view = view.$el;

            // TODO: we shouldn't have to do this
            view.onAttach();
        });

        afterEach(function() {
            layout.getRegion('content').empty();
        });

        it('has a button for the electronic calendar', function() {
            expect($view.find('#add-to-calendar-btn > span:first-child').text()).toEqual('Add to Calendar');
        });
        
        it('has a feedback link', function () {
            expect($view.find('#appointment-feedback-link > a:first-child').attr('href')).toBe('#feedback');
            expect($view.find('#appointment-feedback-link > a:first-child').text()).toEqual('Provide feedback on this submission.');
        });

        describe('when using a Booked Appointment View', function() {
            it('can get the electronic calendar as a string', function() {
                var appointmentsIcs;

                model.set('bookingNote', 'Reason');
                appointmentsIcs = 'BEGIN:VCALENDAR\n' +
                    'PRODID:-//Department of Veterans Affairs//VAR-WEB v3.0//EN\n' +
                    'VERSION:2.0\n' +
                    'BEGIN:VEVENT\n' +
                    'UID:20160611T090000244\n' +
                    'DTSTAMP:' + moment.utc().format('YYYYMMDDTHHmmss') + 'Z\n' +
                    'DTSTART:20160611T090000\n' +
                    'DTEND:20160611T093000\n' +
                    'LOCATION:\n' +
                    'SUMMARY:FACILITY VAMC, ABC Clinic\n' +
                    'DESCRIPTION:\n' +
                    'END:VEVENT\n' +
                    'END:VCALENDAR';
                expect(view.getCalendar()).toEqual(appointmentsIcs);
            });
        });

        describe('when clicking electronic calendar button', function() {
            it('should open an ics file', function() {
                spyOn(window, 'open');
                $view.find('#add-to-calendar-btn').click();
                expect(window.open).toHaveBeenCalledWith('data:text/calendar;charset=utf-8,' + encodeURIComponent(view.getCalendar()));
            });
        });

        describe('when the appointment is not cancelled', function() {
            it('has a primary content header', function() {
                expect($view.find('.primary-header').find('h2')
                    .text()).toEqual('Appointment Details');
            });
            it('shows details date/time, clinic info, and cancel appointment btn', function() {
                var $clinic;
                var $fieldset = $view.find('#appointment-details-section .field-set-flex:eq(0)');
                // Convert <br> to a space to allow .text() to treat it as plain whitespace
                $clinic = $fieldset.find('.field-contain:eq(1)').clone();
                $clinic.find('br').replaceWith(' ');

                expect(cleanUpWhitespace($clinic.text())).toEqual('Details: FriendlyName ABC Clinic Please arrive 30 minutes before your scheduled time unless otherwise directed. VA Facility Locator');

                expect(cleanUpWhitespace($fieldset.find('.field-contain:eq(2)').text())).toEqual('Cancel Appointment');
                $fieldset = $view.find('#appointment-details-section .field-set:eq(0)');
                expect(cleanUpWhitespace($fieldset.find('.field-contain').text())).toEqual('Reason for Appointment: Reason');
            });

            it('shows reason for appointment', function() {
                var $fieldset = $view.find('#appointment-details-section .field-set:eq(0)');

                expect(cleanUpWhitespace($fieldset.find('.field-contain:eq(0)').text())).toEqual('Reason for Appointment: Reason');
            });
        });

        describe('When community care booked appointment exists', function() {
            beforeEach(function() {
                var Model = BookedAppointmentsCollection.prototype.model;
                model = new Model({
                    '_id': '2ce',
                    '_class': 'BookCCAppointment',
                    'appointmentTime': '03/01/2018 07:00:00',
                    'distanceEligibleConfirmed': true,
                    'name': {
                        'firstName': 'benben2',
                        'lastName': 'werner',
                    },
                    'providerPractice': 'schulhouse',
                    'providerPhone': '(234) 567-8900',
                    'address': {
                        'street': '345 Deer Park St',
                        'city': 'Chantilly',
                        'state': 'VA',
                        'zipCode': '23150',
                    },
                    'instructionsToVeteran': 'Zulfi is in charge',
                    'timeZone': '-05:00 EST',
                }, {parse: true});
                layout.getRegion('content').empty();
                ccview = new BookedAppointmentView({model: model});
                layout.showChildView('content', ccview);
                $ccview = ccview.$el;
            });

            it('shows booked community care details.', function() {
                var $fieldset = $ccview.find('#appointment-details-section .field-set-flex:eq(0)');
                // new line was stripped from text
                expect(cleanUpWhitespace($fieldset.find('.field-contain-block:eq(0)').text())).toEqual('Date and Time: Thursday, 03/01/20180700 EST Add to Calendar');
                expect(cleanUpWhitespace($fieldset.find('.field-contain-block:eq(1)').text()))
                    .toEqual('Details: Community Appointment schulhouse benben2 werner 345 Deer Park St Chantilly, VA 23150 (234) 567-8900');
                // new line was stripped from text
                expect(cleanUpWhitespace($fieldset.find('.field-contain-block:eq(2)').text()))
                    .toEqual('Need to cancel or update this appointment?Please contact the provider directly');
            });

            it('can get the electronic calendar as a string', function() {
                var appointmentsIcs;

                model.set('bookingNote', 'Reason');
                appointmentsIcs = 'BEGIN:VCALENDAR\n' +
                    'PRODID:-//Department of Veterans Affairs//VAR-WEB v3.0//EN\n' +
                    'VERSION:2.0\n' +
                    'BEGIN:VEVENT\n' +
                    'UID:20180301T070000schulhouse\n' +
                    'DTSTAMP:' + moment.utc().format('YYYYMMDDTHHmmss') + 'Z\n' +
                    'DTSTART:20180301T070000\n' +
                    'DTEND:20180301T070000\n' +
                    'LOCATION:345 Deer Park St \\nChantilly VA 23150\n' +
                    'SUMMARY:schulhouse\n' +
                    'DESCRIPTION:Zulfi is in charge\n' +
                    'END:VEVENT\n' +
                    'END:VCALENDAR';
                expect(ccview.getCalendar()).toEqual(appointmentsIcs);
            });
        });
    });

    function cleanUpWhitespace(text) {
        var whitespaceRegex = /\s+/g;
        var cleanedText = text.replace(whitespaceRegex, ' ').trim();
        return cleanedText;
    }
});
